﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Windows.Forms;

using KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary; //For search
using KONICAMINOLTA.OpenAPISDK.CommonLibrary.Facade;

namespace OpenAPI_App_Manager
{
    public delegate void ChangedEventHandler(object sender, EventArgs e);
    public delegate void MFPSelectedEventHandler(object sender, MFPConfig mfp);
    public delegate void OpenMFPEventHandler(object sender, MFPConfig mfp);
    public delegate void AdvancedSearchSettingsChangedEventHandler(object sender, AdvancedSearchSettings Settings);

    public partial class MFPList : UserControl
    {
        public event ChangedEventHandler ListChanged;
        public event MFPSelectedEventHandler MFPSelected;
        public event OpenMFPEventHandler OpenMFP;
        public event AdvancedSearchSettingsChangedEventHandler AdvancedSearchSettingsChanged;

        private bool bUpdatingDictionary = false;
        private bool bSelectingIndex = false;

        //Items for our Search Functionality
        private SearchDeviceManager sdMgr;
        private List<MFPConfig> lDiscovered;
        public AppSettings.SearchSettings SearchSettings;
        public AdvancedSearchSettings AdvancedSearch;
        private bool UsingAdvancedSearch;        

        private Dictionary<string, MFPConfig> mfps;
        public List<MFPConfig> MFPs
        {
            get
            {
                List<MFPConfig> lMFPs = new List<MFPConfig>();
                foreach (KeyValuePair<string, MFPConfig> mfp in mfps)
                    lMFPs.Add(mfp.Value);
                return lMFPs;
            }
        }

        public MFPList()
        {
            InitializeComponent();
            mfps = new Dictionary<string, MFPConfig>();
            lDiscovered = new List<MFPConfig>();
            SearchSettings = new AppSettings.SearchSettings();
            UsingAdvancedSearch = false;
            sdMgr = new SearchDeviceManager();
            sdMgr.EventSearchDevice += new SearchDeviceEventHandler(sdMgr_EventSearchDevice);
        }

        #region ListView handling for threads
        private delegate void AddToListViewDelegate(ListViewItem lvi);
        private void AddToListView(ListViewItem lvi)
        {
            listViewMFPs.Items.Add(lvi);
        }
        private void AddItemToListView(ListViewItem lvi)
        {           
            if(this.InvokeRequired)
            this.Invoke(new AddToListViewDelegate(AddToListView), lvi);
            else
                listViewMFPs.Items.Add(lvi);         
        }
        #endregion

        #region Search Button handling for threads
        private delegate void EnableSearchButtonDelegate(bool Enable);
        private void EnableSearchButton(bool Enable)
        {
            buttonSearch.Enabled = Enable;
            buttonSearchAdvanced.Enabled = Enable;
        }
        private void EnableDisableSearch(bool Enable)
        {            
            if (this.InvokeRequired)
            this.Invoke(new EnableSearchButtonDelegate(EnableSearchButton), Enable);
            else
                EnableSearchButton(Enable);         
        }
        #endregion

        private void checkBoxesViews_CheckedChanged(object sender, EventArgs e)
        {
            if (sender == checkBoxDetailView && checkBoxDetailView.Checked)
                checkBoxIconView.Checked = false;
            else if (sender == checkBoxDetailView && !checkBoxDetailView.Checked)
                checkBoxIconView.Checked = true;
            else if (sender == checkBoxIconView && checkBoxIconView.Checked)
                checkBoxDetailView.Checked = false;
            else if (sender == checkBoxIconView && !checkBoxIconView.Checked)
                checkBoxDetailView.Checked = true;

            if (checkBoxDetailView.Checked)
                listViewMFPs.View = View.Details;
            else
                listViewMFPs.View = View.LargeIcon;
        }

        public void Add(MFPConfig MFP)
        {
            if (mfps.ContainsKey(MFP.Address))
                return;

            mfps.Add(MFP.Address, MFP);
            ListViewItem lvi = new ListViewItem(new string[] { 
                MFP.Address,
                MFP.DisplayModel,
                MFP.Port.ToString(),
                MFP.SSL.ToString(),
                MFP.FunctionVersion.Setup.ToStringList(),
                MFP.FunctionVersion.Scan.ToStringList(),
                MFP.FunctionVersion.PullPrint.ToStringList(),
                MFP.FunctionVersion.Authentication.ToStringList(),
                MFP.FunctionVersion.OtherA.ToStringList(),
                MFP.FunctionVersion.ScanWithBrowser.ToStringList(),
                MFP.FunctionVersion.PullPrintWithBrowser.ToStringList(),
                MFP.FunctionVersion.AuthenticationWithBrowser.ToStringList(),
                MFP.FunctionVersion.BrowserLauncher.ToStringList(),
                MFP.FunctionVersion.ScreenSaver.ToStringList(),
                MFP.FunctionVersion.JobManagement.ToStringList(),
                MFP.FunctionVersion.BrowserGeneral.ToStringList()
        });
            lvi.Tag = MFP.Address;
            lvi.ToolTipText = MFP.Model;
            lvi.ImageIndex = GetImageIndexFromModel(MFP);
            AddItemToListView(lvi); //Calling threadsafe version, as this can come from other threads...
        }

        private void buttonAdd_Click(object sender, EventArgs e)
        {            
            bUpdatingDictionary = true;
            string szIPAddress = textBoxAddress.Text;
            //Then add the new dictionary entry for this MFP (if there isn't already another MFP with this address)
            MFPConfig mfp = new MFPConfig();
            mfp.Address = szIPAddress;
            mfp.Password = textBoxPassword.Text;
            mfp.OpenAPIUsername = textBoxOpenAPIUsername.Text;
            mfp.OpenAPIPassword = textBoxOpenAPIPassword.Text;

            ListViewItem lvi = new ListViewItem(new string[] { szIPAddress, "Unknown", "?", "?", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A" });
            lvi.Tag = szIPAddress;
            lvi.ToolTipText = "Unknown";
            lvi.ImageIndex = 8;

            listViewMFPs.Items.Add(lvi);
            listViewMFPs.Items[listViewMFPs.Items.Count - 1].Selected = true;
            listViewMFPs.Select();

            if (szIPAddress == string.Empty)
            {
                listViewMFPs.SelectedItems[0].SubItems[0].Text = "NO ADDRESS";
                bUpdatingDictionary = false;
                return;
            }

            if (!mfps.ContainsKey(mfp.Address))
            {
                mfps.Add(mfp.Address, mfp);
            }
            else
            {
                //There's already another MFP with this address - not only do we not add it to
                //the MFP List, but we warn the user by changing the text in the listbox
                listViewMFPs.SelectedItems[0].SubItems[0].Text = "DUPLICATE ADDRESS";
            }

            QueryMFPInThread(szIPAddress);

            if (ListChanged != null)
                ListChanged(this, e);
            bUpdatingDictionary = false;
            //Deactivate the selected item in listview once mfp added in the list.
            ClearAfterMFPAdd();
        }

        private void ClearAfterMFPAdd()
        {
            listViewMFPs.SelectedItems[0].Selected = false;            
            textBoxAddress.Clear();
            textBoxPassword.Clear();
            textBoxOpenAPIUsername.Clear();
            textBoxOpenAPIPassword.Clear();
            
        }

        private void buttonRemove_Click(object sender, EventArgs e)
        {
            bUpdatingDictionary = true;

            if (listViewMFPs.SelectedItems.Count != 1)
            {
                bUpdatingDictionary = false;
                return;
            }

            if (mfps.ContainsKey(listViewMFPs.SelectedItems[0].SubItems[0].Text))
                mfps.Remove(listViewMFPs.SelectedItems[0].SubItems[0].Text);
            listViewMFPs.Items.Remove(listViewMFPs.SelectedItems[0]);
                                   
            textBoxAddress.Text = string.Empty;
            textBoxPassword.Text = string.Empty;
            textBoxOpenAPIUsername.Text = string.Empty;
            textBoxOpenAPIPassword.Text = string.Empty;

            if (ListChanged != null)
                ListChanged(this, e);

            textBoxAddress.Select();
            buttonAdd.Enabled = true;
            buttonRemove.Enabled = false;
            bUpdatingDictionary = false;
        }

        private void UpdateMFPList(object sender, EventArgs e)
        {
            // MFPList will be updated when editing is enabled
            if (bSelectingIndex || listViewMFPs.SelectedItems.Count <= 0)
                return;

            bUpdatingDictionary = true;

            try
            {
                //If the dictionary already contains the currently selected MFP, we should back up it's data and remove it (we'll add it back again in a second)
                MFPConfig mfpTemp = new MFPConfig();
                mfpTemp.Address = listViewMFPs.SelectedItems[0].SubItems[0].Text;
                if (mfps.ContainsKey(mfpTemp.Address))
                {
                    mfpTemp = mfps[mfpTemp.Address];
                    mfps.Remove(mfpTemp.Address);
                }

                if (textBoxAddress.Text == string.Empty)
                {
                    listViewMFPs.SelectedItems[0].SubItems[0].Text = "NO ADDRESS";
                    bUpdatingDictionary = false;
                    return;
                }

                //While we're here, we'll also update our ListView item in case the address has changed
                listViewMFPs.SelectedItems[0].SubItems[0].Text = textBoxAddress.Text;

                //Then add the new dictionary entry for this MFP (if there isn't already another MFP with this address)
                MFPConfig mfp = mfpTemp;
                mfp.Address = textBoxAddress.Text;
                mfp.Password = textBoxPassword.Text;
                mfp.OpenAPIUsername = textBoxOpenAPIUsername.Text;
                mfp.OpenAPIPassword = textBoxOpenAPIPassword.Text;

                if (!mfps.ContainsKey(mfp.Address))
                {
                    mfps.Add(mfp.Address, mfp);
                }
                else
                {
                    //There's already another MFP with this address - not only do we not add it to
                    //the MFP List, but we warn the user by changing the text in the listbox
                    listViewMFPs.SelectedItems[0].SubItems[0].Text = "DUPLICATE ADDRESS";
                }
            }
            catch
            {
                //This catch can be hit if there was nothing selected in the box when the user hit the remove button (for example)
            }

            if (ListChanged != null)
                ListChanged(this, e);

            bUpdatingDictionary = false;
        }

        private void QueryMFPInThread(object oIPAddress)
        {
            string szAddress = (string)oIPAddress;
            if(!mfps.ContainsKey(szAddress))
                return;

            Result r = mfps[szAddress].QueryMFP();
            if (!r.Success)
            {
                mfps[szAddress] = new MFPConfig();
                mfps[szAddress].Address = szAddress;
            }

            UpdateSelectedMFP(listViewMFPs.SelectedItems);
        }

        private delegate void UpdateSelectedMFPDelegate(ListView.SelectedListViewItemCollection selectedListViewItemCollection);
        private void UpdateSelectedMFP(ListView.SelectedListViewItemCollection selectedListViewItemCollection)
        {
            if(this.InvokeRequired)
              this.Invoke(new UpdateSelectedMFPDelegate(UpdateSelectedMFPInThread), selectedListViewItemCollection);
            else
              UpdateSelectedMFPInThread(selectedListViewItemCollection);
          
        }
        private void UpdateSelectedMFPInThread(ListView.SelectedListViewItemCollection selectedListViewItemCollection)
        {
            for (int i = 0; i < selectedListViewItemCollection.Count; i++)
            {
                if (!this.mfps.ContainsKey(selectedListViewItemCollection[i].SubItems[0].Text))
                    break;
                MFPConfig mfp = this.mfps[selectedListViewItemCollection[i].SubItems[0].Text];

                selectedListViewItemCollection[i].SubItems[0].Text = mfp.Address;
                selectedListViewItemCollection[i].SubItems[1].Text = mfp.DisplayModel;
                selectedListViewItemCollection[i].SubItems[2].Text = mfp.Port.ToString();
                selectedListViewItemCollection[i].SubItems[3].Text = mfp.SSL.ToString();
                selectedListViewItemCollection[i].SubItems[4].Text = mfp.FunctionVersion.Setup.ToStringList();
                selectedListViewItemCollection[i].SubItems[5].Text = mfp.FunctionVersion.Scan.ToStringList();
                selectedListViewItemCollection[i].SubItems[6].Text = mfp.FunctionVersion.PullPrint.ToStringList();
                selectedListViewItemCollection[i].SubItems[7].Text = mfp.FunctionVersion.Authentication.ToStringList();
                selectedListViewItemCollection[i].SubItems[8].Text = mfp.FunctionVersion.OtherA.ToStringList();
                selectedListViewItemCollection[i].SubItems[9].Text = mfp.FunctionVersion.ScanWithBrowser.ToStringList();
                selectedListViewItemCollection[i].SubItems[10].Text = mfp.FunctionVersion.PullPrintWithBrowser.ToStringList();
                selectedListViewItemCollection[i].SubItems[11].Text = mfp.FunctionVersion.AuthenticationWithBrowser.ToStringList();
                selectedListViewItemCollection[i].SubItems[12].Text = mfp.FunctionVersion.BrowserLauncher.ToStringList();
                selectedListViewItemCollection[i].SubItems[13].Text = mfp.FunctionVersion.ScreenSaver.ToStringList();
                selectedListViewItemCollection[i].SubItems[14].Text = mfp.FunctionVersion.JobManagement.ToStringList();
                selectedListViewItemCollection[i].SubItems[15].Text = mfp.FunctionVersion.BrowserGeneral.ToStringList();

                selectedListViewItemCollection[i].Tag = mfp.Address;
                selectedListViewItemCollection[i].ToolTipText = mfp.Model;
                selectedListViewItemCollection[i].ImageIndex = GetImageIndexFromModel(mfp);
            }
        }

        private int GetImageIndexFromModel(MFPConfig mfp)
        {
            string szModelCheck = mfp.Model.ToUpper().Replace(" ", "");
            bool bColour = mfp.ColourPrint;
            bool bKM = (szModelCheck.Contains("KONICAMINOLTA"));
            bool bProduction = (szModelCheck.Contains("PRESS") || szModelCheck.Contains("PRO"));
            if (szModelCheck == "UNKNOWN")
                return 8;
            if (szModelCheck.Contains("MAGICOLOR"))
                return 5; //Magicolor
            if ((szModelCheck.Contains("BIZHUBC35")) && (!szModelCheck.Contains("BIZHUBC353")))
                return 1; //C35
            if ((szModelCheck.Contains("BIZHUB")) && (szModelCheck.EndsWith("P")))
                return 2; //bizhub Printer

            if (bKM && bColour && bProduction)
                return 7; //Colour, KM, Production
            else if (bKM && bProduction)
                return 6; //Black White, KM, Production
            else if (bKM && bColour)
                return 0; //Colour, KM
            else if (bColour)
                return 4; //Colour, Generic
            else if (bKM)
                return 3; //Black White, KM
            else
                return 3; //Black White, Generic
        }

        private void listViewMFPs_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (!bUpdatingDictionary)
            {
                bSelectingIndex = true;

                if (listViewMFPs.SelectedItems.Count == 0)
                {
                    //Enable the add button when item is selected in list
                    buttonAdd.Enabled = true;
                    buttonRemove.Enabled = false;
                    textBoxAddress.Text = string.Empty;
                    textBoxPassword.Text = string.Empty;
                    textBoxOpenAPIUsername.Text = string.Empty;
                    textBoxOpenAPIPassword.Text = string.Empty;                
                   
                }
                else
                {                                 
                    //Disable the add button when no item is selected in list                   
                    buttonAdd.Enabled = false;
                    buttonRemove.Enabled = true;
                    if (mfps.ContainsKey(listViewMFPs.SelectedItems[0].SubItems[0].Text))
                    {
                        //If this is an item that's already in our dictionary, we populate
                        //the UI elements from the dictionary
                        MFPConfig mfp = mfps[listViewMFPs.SelectedItems[0].SubItems[0].Text];
                        textBoxAddress.Text = mfp.Address;
                        textBoxPassword.Text = mfp.Password;
                        textBoxOpenAPIUsername.Text = mfp.OpenAPIUsername;
                        textBoxOpenAPIPassword.Text = mfp.OpenAPIPassword;

                        if (MFPSelected != null)
                            MFPSelected(this, mfp);
                    }
                    else
                    {
                        //It's a new MFP
                        textBoxAddress.Text = string.Empty;
                        textBoxPassword.Text = "12345678";
                        textBoxOpenAPIUsername.Text = string.Empty;
                        textBoxOpenAPIPassword.Text = string.Empty;
                    }
                }

                // Update the MFP settings in xml file
                if (ListChanged != null)
                    ListChanged(this, e);

                bSelectingIndex = false;
            }
        }

        private void listViewMFPs_DoubleClick(object sender, EventArgs e)
        {
            if (listViewMFPs.SelectedItems.Count != 1)
                return;

            string szAddress=listViewMFPs.SelectedItems[0].SubItems[0].Text;
            if (mfps.ContainsKey(szAddress))
            {
                if (OpenMFP != null)
                    OpenMFP(this, mfps[szAddress]);
            }
        }

        #region OpenAPI Search Functionality
        private void buttonSearch_Click(object sender, EventArgs e)
        {
            UsingAdvancedSearch = false;
            PerformSearch();
        }

        private void buttonSearchAdvanced_Click(object sender, EventArgs e)
        {
            frmAdvancedSearch winSearch = new frmAdvancedSearch(this.AdvancedSearch);
            if (winSearch.ShowDialog() == DialogResult.OK)
            {
                this.AdvancedSearch = winSearch.Settings;
                 
                this.listViewMFPs.Items.Clear();
                this.mfps.Clear();

                if (AdvancedSearchSettingsChanged != null)
                    AdvancedSearchSettingsChanged(this, this.AdvancedSearch);

                //Perform Search
                UsingAdvancedSearch = true;
                PerformSearch();
            }
        }

        public void PerformSearch()
        {
            //Handle the UI
            EnableDisableSearch(false);

            //Clear our "Discovered" list
            lDiscovered.Clear();

            //Create a SearchDeviceSetting to set up the details of the search
            KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.SearchDeviceSetting searchDeviceSetting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.SearchDeviceSetting();

            if (SearchSettings.SearchMethod == "UPnP")
            {
                //UPnP search is the most common form, and is recommended for most applications due to the ease and reliability of using it.
                searchDeviceSetting.Protocol = 1;                             //1 = UPnP
                searchDeviceSetting.UpnpSetting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.UpnpSetting();
                searchDeviceSetting.UpnpSetting.Timeout = 10000;              //10 seconds
                searchDeviceSetting.UpnpSetting.Ttl = 0;                      //No TTL (unlimited)
                searchDeviceSetting.UpnpSetting.IsSearchTargetDefault = true; //Default search target - should be no good reason to change this
                searchDeviceSetting.UpnpSetting.IsIPv6 = false;               //We're only doing IPv4 searches
                searchDeviceSetting.UpnpSetting.IsSiteLocal = false;          //Only used for IPv6
            }
            else
            {
                //"SNMPBroadcast" or "SNMPPing"
                searchDeviceSetting.Protocol = 2; //2 = SNMP (Broadcast or Ping)
                searchDeviceSetting.SnmpSetting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.SnmpSetting();
                searchDeviceSetting.SnmpSetting.Port = 161; //Standard SNMP port
                searchDeviceSetting.SnmpSetting.RetryCount = 3; //Number of retries
                searchDeviceSetting.SnmpSetting.SessionTime = 5000; //Timeout - 5 seconds
                if (SearchSettings.SNMPVersion == 1)
                {
                    searchDeviceSetting.SnmpSetting.SnmpVerison = 1; //1 = SNMPv1
                    searchDeviceSetting.SnmpSetting.Verison1Setting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.Verison1Setting();
                    searchDeviceSetting.SnmpSetting.Verison1Setting.Community = SearchSettings.SNMPv1Community;
                }
                else //It's SNMPv3
                {
                    searchDeviceSetting.SnmpSetting.SnmpVerison = 2; //2 = SNMPv3
                    searchDeviceSetting.SnmpSetting.Verison3Setting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.Verison3Setting();
                    searchDeviceSetting.SnmpSetting.Verison3Setting.UserName = SearchSettings.SNMPv3Username;
                    searchDeviceSetting.SnmpSetting.Verison3Setting.ContextName = SearchSettings.SNMPv3Context;
                }
                if (SearchSettings.SearchMethod == "SNMPBroadcast")
                {
                    searchDeviceSetting.SnmpSetting.SearchType = 1; //1 = SNMP Broadcast
                    searchDeviceSetting.SnmpSetting.BroadcastSetting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.BroadcastSetting();
                    searchDeviceSetting.SnmpSetting.BroadcastSetting.BroadcastList = new string[] { SearchSettings.SNMPBroadcastAddress }; //You could have a list of broadcast addresses, we're only using one
                    searchDeviceSetting.SnmpSetting.BroadcastSetting.BroadCastTime = 5000; //Timeout - 5 seconds
                }

                else //It's "SNMPPing"
                {
                    searchDeviceSetting.SnmpSetting.SearchType = 2; //2 = SNMP Ping for IPv4 (3 = SNMP Ping for IPv6, not implemented here)
                    searchDeviceSetting.SnmpSetting.PingSetting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.SearchDeviceSettingNS.PingSetting();
                    searchDeviceSetting.SnmpSetting.PingSetting.StartIPAddress = SearchSettings.SNMPPingStartAddress;
                    searchDeviceSetting.SnmpSetting.PingSetting.EndIPAddress = SearchSettings.SNMPPingEndAddress;
                }
            }

            KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.ResultStatusNS.ResultStatus result = sdMgr.SearchDevice(searchDeviceSetting);
            if (result == null || result.ResultStatusInfo != "1")
            {
                MessageBox.Show("Search failed to start successfully.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                EnableDisableSearch(true);
            }
        }

        private bool CheckFVWithOperator(string fvList)
        {
            string[] arrFV = fvList.Split(',');
            bool result = false;
            double versn;
            // Adanced Search using the one of the following search Search functionlities.
            try
            {
                switch (this.AdvancedSearch.FunctionVersionOperator)
                {
                    case "Has":
                        //Searches the MFP which has the specified function version
                        if (fvList.Contains(this.AdvancedSearch.FunctionVersionAppVersion))
                        {
                            result = true;
                        }
                        break;
                    case ">":
                        //Searches the MFP which has the function version greater than the specified 
                        // function version
                        foreach (string ver in arrFV)
                        {
                            versn = double.Parse(ver);
                            if (versn > double.Parse(this.AdvancedSearch.FunctionVersionAppVersion))
                            {
                                result = true;
                                break;
                            }
                        }
                        break;
                    case "<":
                        //Searches the MFP which has the function version less than the specified 
                        // function version
                        foreach (string ver in arrFV)
                        {
                            versn = double.Parse(ver);
                            if (versn < double.Parse(this.AdvancedSearch.FunctionVersionAppVersion))
                            {
                                result = true;
                                break;
                            }
                        }
                        break;
                    case ">=":
                        //Searches the MFP which has the function version equal and greater than the  
                        // specified function version
                        foreach (string ver in arrFV)
                        {
                            versn = double.Parse(ver);
                            if (versn >= double.Parse(this.AdvancedSearch.FunctionVersionAppVersion))
                            {
                                result = true;
                                break;
                            }
                        }
                        break;
                    case "<=":
                        //Searches the MFP which has the function version less and greater than the  
                        // specified function version
                        foreach (string ver in arrFV)
                        {
                            versn = double.Parse(ver);
                            if (versn <= double.Parse(this.AdvancedSearch.FunctionVersionAppVersion))
                            {
                                result = true;
                                break;
                            }
                        }
                        break;
                    case "!=":
                        //Searches the MFP which has the function version which does not have the  
                        // specified function version
                        foreach (string ver in arrFV)
                        {
                            versn = double.Parse(ver);
                            if (versn == double.Parse(this.AdvancedSearch.FunctionVersionAppVersion))
                            {
                                result = false;
                                break;
                            }                            
                        }
                        break;
                    case "Between":
                        //Searches the MFP which has the function version which is between the specified 
                        // specified function version
                        foreach (string ver in arrFV)
                        {
                            versn = double.Parse(ver);
                            if (versn >= double.Parse(this.AdvancedSearch.FunctionVersionAppVersion) && versn <= double.Parse(this.AdvancedSearch.FunctionVersionAppVersion1))
                            {
                                result = true;
                                break;
                            }
                        }
                        break;
                }
            }
            catch (System.FormatException)
            {
                result = false;
            }
            return result;
        }

        private object searchLock = new object();  //This lock ensures that once a device is found, it gets added to the list before the "ended" actions take place
        private void sdMgr_EventSearchDevice(object sender, SearchDeviceEventArgs e)
        {
            //This event will continue until EndFlg is hit, which is the natural end of the search, or an error.
            if (!e.EndFlg)
            {
                lock (searchLock)
                {
                    //This is not the end of the search, so that means we must have found a device.  We want to add it to our "Discovered" list.
                    MFPConfig mfp = new MFPConfig();                    

                    mfp.Address = e.SearchDeviceInfo.IPAddress;
                    mfp.QueryMFP();
                    bool bOkayToAdd = true;                    
                   
                    if (UsingAdvancedSearch)
                    {
                        if (this.AdvancedSearch.ProductName && !mfp.Model.Contains(this.AdvancedSearch.ProductNameText))
                            bOkayToAdd = false;
                        
                        if (bOkayToAdd && this.AdvancedSearch.FunctionVersion)
                        {
                            switch (this.AdvancedSearch.FunctionVersionAppType)
                            {
                                case "Scan":                                                                     
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "PullPrint":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "JobManagement":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "Authentication":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "Setup":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "ScanBrowser":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "PullPrintBrowser":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "AuthenticationBrowser":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "ScreenSaver":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "BrowserLauncher":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "OtherA":                                    
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;

                                case "BrowserGeneral":
                                    bOkayToAdd = CheckFVWithOperator(mfp.FunctionVersion.Scan.ToStringList());
                                    break;
                            }
                        }

                        // To search using Application name and Color
                        OpenAPIWrapper openAPIHandler;
                        KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult cr;
                        if (bOkayToAdd && this.AdvancedSearch.Application)
                        {
                            try
                            {
                                openAPIHandler = new OpenAPIWrapper(mfp);
                                KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationInfoNS.ApplicationInfoCollection AppInfos = new KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationInfoNS.ApplicationInfoCollection();
                                cr = openAPIHandler.GetApplicationList(out AppInfos);
                                if (cr != KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult.Ack)
                                    bOkayToAdd = false;
                                else
                                {
                                    bOkayToAdd = false;
                                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationInfoNS.ApplicationInfo app in AppInfos)
                                    {
                                        if (app.Name.ToUpper() == this.AdvancedSearch.ApplicationName.ToUpper())
                                        {
                                            bOkayToAdd = true;
                                            break;
                                        }
                                    }
                                }
                            }
                            catch
                            {
                                bOkayToAdd = false;
                            }
                        }

                        //Searching with selected SSL option
                        if (bOkayToAdd && this.AdvancedSearch.SSLSetting)
                        {
                            if (mfp.SSL && (this.AdvancedSearch.SSLText == "True"))
                            {
                                bOkayToAdd = true;
                            }
                            else if (!mfp.SSL && (this.AdvancedSearch.SSLText == "False"))
                            {
                                bOkayToAdd = true;
                            }                          
                            else
                            {
                                bOkayToAdd = false;
                            }
                        }
                                               
                        // Searching with selected color option
                        if (this.AdvancedSearch.ColorSetting)
                        {
                            string colorval = "";
                            if (this.AdvancedSearch.ColorText == "Color")
                            {
                                colorval = "1";
                            }
                            else if (this.AdvancedSearch.ColorText == "Monochrome")
                            {
                                colorval = "0";
                            }

                            // Gets DeviceInfo to check with color option
                            openAPIHandler = new OpenAPIWrapper(mfp);
                            cr = openAPIHandler.GetDeviceInfo();                            

                            if (cr == KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult.Ack)
                            {
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.Capability != null &&
                                    colorval == openAPIHandler.SetupMainControlManager.DeviceInfo.Capability.Color)
                                    bOkayToAdd = true;
                                else
                                    bOkayToAdd = false;
                            }
                            else
                            {
                                bOkayToAdd = false;
                            }
                        }                                                                     
                    }

                    //Add the items to 'Discovered' list
                    if (bOkayToAdd)
                    {
                        lDiscovered.Add(mfp);
                    }                       
                }
            }
            else
            {
                lock (searchLock)
                {
                    //Our search is finished, we need to re-enable the search button
                    EnableDisableSearch(true);
                    //And add the discovered items to our ListView
                    foreach (MFPConfig mfp in lDiscovered)
                        Add(mfp);

                    //Trigger our event
                    if (ListChanged != null)
                        ListChanged(this, e);
                }
            }
        }
        #endregion
            

        private void textBoxAddress_Leave(object sender, EventArgs e)
        {
            if (listViewMFPs.SelectedItems.Count != 1)
                return;

            UpdateSelectedMFP(listViewMFPs.SelectedItems);
            string szIPAddress = textBoxAddress.Text;
            if (!mfps.ContainsKey(szIPAddress))
                return;

            System.Net.IPAddress ip = null;
            if (System.Net.IPAddress.TryParse(szIPAddress, out ip))
            {
                Thread tQuery = new Thread(new ParameterizedThreadStart(QueryMFPInThread));
                tQuery.Start(szIPAddress);
            }
        }      
    }
}
